/* eslint-disable @typescript-eslint/no-explicit-any */
/// <reference lib="dom" />
import { usePage } from "@inertiajs/react";
import { useBluetooth } from "@/contexts/bluetooth-context";
import { DropdownMenuItem } from "@/components/ui/dropdown-menu";
import { Bluetooth } from "lucide-react";
import { toast } from "sonner";

interface NotaBtProps {
    tx: any;
    paperWidth?: number;
}

export default function NotaBt({ tx, paperWidth = 32 }: NotaBtProps) {
    const { printerServer, isBluetoothConnected } = useBluetooth();

    const centerText = (text: string, width = paperWidth) => {
        if (!text) return "";
        if (text.length >= width) return text;
        const leftPadding = Math.floor((width - text.length) / 2);
        return " ".repeat(leftPadding) + text;
    };

    const formatLine = (left: string, right: string, width = paperWidth) => {
        const leftText = left.trim();
        const rightText = String(right).trim();

        const total = leftText.length + rightText.length;
        const spaces = " ".repeat(Math.max(width - total, 2));

        return `${leftText}${spaces}${rightText}`;
    };


    const writeInChunks = async (
        characteristic: BluetoothRemoteGATTCharacteristic,
        data: Uint8Array,
        chunkSize = 200
    ) => {
        for (let i = 0; i < data.length; i += chunkSize) {
            const chunk = data.slice(i, i + chunkSize);
            await characteristic.writeValue(chunk);
            await new Promise((resolve) => setTimeout(resolve, 50));
        }
    };

    const handleBluetoothPrintReceipt = async () => {
        if (!printerServer || !isBluetoothConnected) {
            toast.error("❌ Printer Bluetooth belum terhubung");
            return;
        }

        // Ambil data real dari server (produk + varian lengkap)
        const res = await fetch(`/transactions/${tx.id}/nota-bt`);
        const { store, trx } = await res.json();

        try {
            const encoder = new TextEncoder();
            const now = new Date().toLocaleString("id-ID", {
                day: "2-digit",
                month: "2-digit",
                year: "numeric",
                hour: "2-digit",
                minute: "2-digit",
            });

            // ===========================
            // HEADER
            // ===========================
            const headerInfo = [
                formatLine("Invoice", trx.invoice ?? "-"),
                formatLine("Tanggal", now),
                formatLine("Jenis", trx.order_type_label ?? "-"),
                trx.table?.number ? formatLine("Meja", String(trx.table.number)) : null,
                formatLine("Pelanggan", trx.customer?.name ?? trx.customer_name ?? "Umum"),
                formatLine("Kasir", trx.cashier_shift?.user?.name ?? "-")
            ]
                .filter(Boolean)
                .join("\n");


            // ===========================
            // ITEM LIST (PAKAI DATA REAL trx)
            // ===========================
            const itemsText = trx.transaction_details
                .map((it: any) => {
                    const baseName =
                        it.items?.product?.name ??
                        it.items?.name ??
                        "Produk";

                    let variant = "";
                    if (it.items?.product_variant_values?.length > 0) {
                        variant = "[" +
                            it.items.product_variant_values
                                .map((v: any) =>
                                    `${v.variant_value?.variant_option?.name}: ${v.variant_value?.name}`
                                )
                                .join(", ") +
                            "]";
                    }

                    const fullName = `${baseName} ${variant}`.trim();
                    const nameLines = fullName.match(new RegExp(`.{1,${paperWidth}}`, "g")) || [fullName];

                    const qty = Number(it.quantity ?? 1);
                    const price = Number(it.price ?? 0);
                    const discount = Number(it.discount ?? 0);
                    const discountType = it.discount_type ?? "nominal";

                    // ✅ Diskon per item
                    const discountPerItem = discountType === "percentage"
                        ? Math.floor(price * discount / 100)
                        : discount;

                    // ✅ Harga akhir per item
                    const finalPrice = Math.max(0, price - discountPerItem);

                    // ✅ Total item
                    const total = finalPrice * qty;

                    const qtyPrice = formatLine(
                        `${qty} x ${finalPrice.toLocaleString("id-ID")}`,
                        total.toLocaleString("id-ID"),
                        paperWidth
                    );

                    return [...nameLines, qtyPrice].join("\n");
                })
                .join("\n");

            // ===========================
            // TOTAL
            // ===========================
            const subtotal = Number(tx.subtotal ?? 0);
            const discount = Number(tx.discount ?? 0);
            const total = Number(String(tx.grand_total).replace(/[.,]/g, "")) || subtotal;

            // Pajak & Operasional dari backend
            const taxRows = trx.tax_rows || [];
            const ppnAmount = Number(trx.ppn_amount || 0);
            const oprAmount = Number(trx.opr_amount || 0);

            const summary = [
                formatLine("Subtotal", subtotal.toLocaleString("id-ID")),
                discount ? formatLine("Diskon", `-${discount.toLocaleString("id-ID")}`) : null,
                ...taxRows.map((r: any) => formatLine(r.name, r.value.toLocaleString("id-ID"))),
                formatLine("Total", total.toLocaleString("id-ID")),
            ].filter(Boolean).join("\n");


            const payment = [
                formatLine(`Bayar (${tx.payment_method ?? "-"})`, tx.pay ?? "-"),
                formatLine("Kembali", tx.change ?? "-"),
            ].join("\n");

            // ===========================
            // STRUK FINAL
            // ===========================
            const struk = `
${centerText(store.name?.toUpperCase() || "")}
${centerText(store.addr || "")}
${store.phone ? centerText("Telp: " + store.phone) : ""}
${"-".repeat(paperWidth)}
${headerInfo}
${"-".repeat(paperWidth)}
${itemsText}
${"-".repeat(paperWidth)}
${summary}
${"-".repeat(paperWidth)}
${payment}
${"-".repeat(paperWidth)}
${centerText("Terima kasih - Simpan struk ini")}
${centerText("sebagai bukti transaksi.")}
\r\n\r\n`;

            // ===========================
            // KIRIM KE PRINTER
            // ===========================
            const data = encoder.encode(struk);
            const services = await printerServer.getPrimaryServices();

            for (const service of services) {
                const characteristics = await service.getCharacteristics();
                for (const c of characteristics) {
                    if (c.properties.write || c.properties.writeWithoutResponse) {
                        await writeInChunks(c, data);
                        toast.success("Struk berhasil dikirim!");
                        return;
                    }
                }
            }

            toast.error("Printer tidak mendukung mode write");

        } catch (err: any) {
            console.error(err);
            toast.error("Gagal Cetak: " + err.message);
        }
    };


    return (
        <DropdownMenuItem
            onClick={isBluetoothConnected ? handleBluetoothPrintReceipt : undefined}
            disabled={!isBluetoothConnected}
        >
            <Bluetooth /> Cetak Nota Bluetooth
        </DropdownMenuItem>
    );
}
